// Studio Rental Website - Version 3 (HTML/CSS/JS)

const studioTypes = [
  { 
    id: 'backdrop', 
    name: 'Студия с фоном', 
    icon: 'icons/type-backdrop.svg',
    color: '#FF6B35',
    description: 'Профессиональные фоны и освещение',
    gradient: 'linear-gradient(135deg, #FF6B35 0%, #F7931E 100%)',
    price: 'от 2500₽/час',
    area: '50 м²'
  },
  { 
    id: 'chromakey', 
    name: 'Хромакей', 
    icon: 'icons/type-chromakey.svg',
    color: '#F7931E',
    description: 'Зеленая стена для видеосъемки',
    gradient: 'linear-gradient(135deg, #F7931E 0%, #FF6B35 100%)',
    price: 'от 3000₽/час',
    area: '60 м²'
  },
  { 
    id: 'interior', 
    name: 'Интерьер', 
    icon: 'icons/type-interior.svg',
    color: '#FF8C42',
    description: 'Уютные интерьерные пространства',
    gradient: 'linear-gradient(135deg, #FF8C42 0%, #FF6B35 100%)',
    price: 'от 2800₽/час',
    area: '45 м²'
  },
  { 
    id: 'openspace', 
    name: 'Open Space', 
    icon: 'icons/type-openspace.svg',
    color: '#FF6B35',
    description: 'Большое открытое пространство',
    gradient: 'linear-gradient(135deg, #FF6B35 0%, #F7931E 100%)',
    price: 'от 3500₽/час',
    area: '80 м²'
  }
]

const studios = [
  { name: 'StudioPro Premium', location: 'Центр', rating: 5, types: 'Все типы', capacity: 'VIP-студии' },
  { name: 'StudioPro Chroma', location: 'Тверская', rating: 5, types: 'Хромакей, Фон', capacity: '6 студий' },
  { name: 'StudioPro Interior', location: 'Арбат', rating: 5, types: 'Интерьер, Open Space', capacity: '4 студии' },
  { name: 'StudioPro Creative', location: 'Сокольники', rating: 5, types: 'Все типы', capacity: '8 студий' }
]

const advantages = [
  { icon: 'icons/stat-bookings.svg', title: 'Профессиональное оборудование', text: 'Только лучшая техника' },
  { icon: 'icons/stat-studios.svg', title: 'Быстрое бронирование', text: 'Забронируйте за минуты' },
  { icon: 'icons/stat-equipment.svg', title: 'Гибкий график', text: 'Работаем 24/7' },
  { icon: 'icons/stat-rating.svg', title: 'Высокий рейтинг', text: '98% довольных клиентов' }
]

let currentSlide = 0
let slides = []
let dots = []
let slideInterval

function initPage() {
  const currentPage = window.location.pathname.includes('book.html') ? 'book' : 'index'
  
  if (currentPage === 'book') {
    initBookPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadBookings()
  updateStats()
  
  setInterval(() => {
    loadBookings()
    updateStats()
  }, 500)
}

function loadBookings() {
  const saved = localStorage.getItem('studioBookings')
  let bookings = []
  
  if (saved) {
    try {
      bookings = JSON.parse(saved)
    } catch (e) {
      bookings = []
    }
  } else {
    bookings = [
      {
        id: 1,
        studio: 'StudioPro Premium',
        date: '2025-03-20',
        equipment: 'Профессиональное освещение, камеры',
        review: 'Отличная студия, все оборудование работает идеально!'
      },
      {
        id: 2,
        studio: 'StudioPro Chroma',
        date: '2025-03-15',
        equipment: 'Хромакей, студийное освещение',
        review: 'Прекрасные условия для съемки'
      },
      {
        id: 3,
        studio: 'StudioPro Interior',
        date: '2025-03-10',
        equipment: 'Интерьерная студия, естественный свет',
        review: 'Очень уютная атмосфера'
      }
    ]
    localStorage.setItem('studioBookings', JSON.stringify(bookings))
  }
  
  renderBookings(bookings)
  return bookings
}

function renderBookings(bookings) {
  const bookingsGrid = document.getElementById('bookingsGrid')
  if (!bookingsGrid) return
  
  if (bookings.length === 0) {
    bookingsGrid.innerHTML = `
      <div class="empty-state">
        <div class="empty-icon-circle">
          <img src="icons/logo-camera.svg" alt="" width="80" height="80">
        </div>
        <h3 class="empty-title">Нет бронирований</h3>
        <p class="empty-text">Забронируйте студию для вашего проекта прямо сейчас</p>
      </div>
    `
  } else {
    bookingsGrid.innerHTML = bookings.map(booking => {
      const stars = Array.from({ length: 5 }, () => '★').join('')
      
      return `
        <div class="booking-card">
          <div class="booking-header">
            <div class="booking-studio-badge" style="background: rgba(255, 107, 53, 0.25); color: #FF6B35;">
              ${booking.studio}
            </div>
            <div class="booking-rating">
              ${stars}
            </div>
          </div>
          <div class="booking-content">
            <div class="booking-info-item">
              <img src="icons/booking-calendar.svg" alt="Calendar" width="24" height="24">
              <span class="booking-info-text">
                ${new Date(booking.date).toLocaleDateString('ru-RU', { 
                  day: 'numeric', 
                  month: 'long', 
                  year: 'numeric' 
                })}
              </span>
            </div>
            <div class="booking-info-item">
              <img src="icons/booking-video.svg" alt="Video" width="24" height="24">
              <span class="booking-info-text">${booking.equipment}</span>
            </div>
            ${booking.review ? `
              <div class="booking-review">
                <img src="icons/booking-message.svg" alt="Message" width="20" height="20">
                <span class="booking-review-text">${booking.review}</span>
              </div>
            ` : ''}
          </div>
        </div>
      `
    }).join('')
  }
}

function updateStats() {
  const bookings = loadBookings()
  const stats = [
    { 
      value: bookings.length, 
      label: 'Бронирований',
      icon: 'icons/stat-bookings.svg',
      color: 'rgba(255, 107, 53, 0.2)'
    },
    { 
      value: 4.8, 
      label: 'Средняя оценка',
      icon: 'icons/stat-rating.svg',
      color: 'rgba(247, 147, 30, 0.2)'
    },
    { 
      value: new Set(bookings.map(b => b.studio)).size, 
      label: 'Студий',
      icon: 'icons/stat-studios.svg',
      color: 'rgba(255, 140, 66, 0.2)'
    }
  ]
  
  const statsGrid = document.getElementById('statsGrid')
  if (statsGrid) {
    statsGrid.innerHTML = stats.map(stat => `
      <div class="stat-card">
        <div class="stat-icon-wrapper" style="background: ${stat.color};">
          <img src="${stat.icon}" alt="" width="48" height="48">
        </div>
        <div class="stat-content">
          <div class="stat-value">${stat.value}</div>
          <div class="stat-label">${stat.label}</div>
        </div>
      </div>
    `).join('')
  }
}

function initBookPage() {
  const bookingForm = document.getElementById('bookingForm')
  const typesGrid = document.getElementById('typesGrid')
  const dateInput = document.getElementById('date')
  const successMessage = document.getElementById('successMessage')
  const advantagesGrid = document.getElementById('advantagesGrid')
  const studiosGrid = document.getElementById('studiosGrid')

  let selectedType = null

  if (typesGrid) {
    typesGrid.innerHTML = studioTypes.map(type => `
      <button type="button" class="type-card" data-id="${type.id}" style="border-color: rgba(255, 107, 53, 0.4);">
        <div class="type-icon-wrapper" style="background: ${type.color}30;">
          <img src="${type.icon}" alt="${type.name}" width="52" height="52" style="filter: drop-shadow(0 0 12px ${type.color}80);">
        </div>
        <div class="type-content">
          <h3 class="type-name">${type.name}</h3>
          <p class="type-description">${type.description}</p>
          <div class="type-meta">
            <span class="type-area">${type.area}</span>
            <span class="type-price">${type.price}</span>
          </div>
        </div>
      </button>
    `).join('')

    typesGrid.querySelectorAll('.type-card').forEach(button => {
      button.addEventListener('click', function() {
        typesGrid.querySelectorAll('.type-card').forEach(btn => {
          btn.classList.remove('selected')
          btn.style.borderColor = 'rgba(255, 107, 53, 0.4)'
          const check = btn.querySelector('.type-check')
          if (check) check.remove()
        })
        this.classList.add('selected')
        this.style.borderColor = studioTypes.find(t => t.id === this.dataset.id).color
        selectedType = studioTypes.find(t => t.id === this.dataset.id)
        
        const check = document.createElement('div')
        check.className = 'type-check'
        check.style.background = selectedType.gradient
        check.innerHTML = `<img src="icons/btn-check.svg" alt="" width="32" height="32">`
        this.appendChild(check)
      })
    })
  }

  if (advantagesGrid) {
    advantagesGrid.innerHTML = advantages.map(advantage => `
      <div class="advantage-card">
        <div class="advantage-icon">
          <img src="${advantage.icon}" alt="" width="44" height="44">
        </div>
        <h3 class="advantage-title">${advantage.title}</h3>
        <p class="advantage-text">${advantage.text}</p>
      </div>
    `).join('')
  }

  if (studiosGrid) {
    studiosGrid.innerHTML = studios.map(studio => {
      const stars = Array.from({ length: 5 }, () => '★').join('')
      
      return `
        <div class="studio-card">
          <div class="studio-avatar">
            <img src="icons/studio-camera.svg" alt="" width="44" height="44">
          </div>
          <h3 class="studio-name">${studio.name}</h3>
          <p class="studio-location">${studio.location}</p>
          <div class="studio-rating">
            ${stars}
          </div>
          <p class="studio-types">${studio.types}</p>
          <div class="studio-capacity">
            <img src="icons/studio-users.svg" alt="" width="18" height="18">
            <span>${studio.capacity}</span>
          </div>
        </div>
      `
    }).join('')
  }

  if (dateInput) {
    const minDate = new Date().toISOString().split('T')[0]
    const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]
    dateInput.setAttribute('min', minDate)
    dateInput.setAttribute('max', maxDate)
  }

  if (bookingForm) {
    bookingForm.addEventListener('submit', function(e) {
      e.preventDefault()
      if (selectedType && dateInput.value) {
        const savedBookings = localStorage.getItem('studioBookings')
        const bookings = savedBookings ? JSON.parse(savedBookings) : []
        
        const randomStudio = studios[Math.floor(Math.random() * studios.length)]
        const newBooking = {
          id: Date.now(),
          studio: randomStudio.name,
          date: dateInput.value,
          equipment: selectedType.description,
          review: ''
        }

        bookings.push(newBooking)
        localStorage.setItem('studioBookings', JSON.stringify(bookings))
        
        successMessage.style.display = 'flex'
        setTimeout(() => {
          successMessage.style.display = 'none'
          selectedType = null
          dateInput.value = ''
          typesGrid.querySelectorAll('.type-card').forEach(btn => {
            btn.classList.remove('selected')
            btn.style.borderColor = 'rgba(255, 107, 53, 0.4)'
            const check = btn.querySelector('.type-check')
            if (check) check.remove()
          })
          window.location.href = 'index.html'
        }, 2000)
      }
    })
  }
}

function initSlider() {
  const sliderContainer = document.getElementById('slider')
  if (!sliderContainer) return

  slides = Array.from(sliderContainer.querySelectorAll('.slide'))
  dots = Array.from(sliderContainer.querySelectorAll('.dot'))
  
  const sliderPrev = document.getElementById('sliderPrev')
  const sliderNext = document.getElementById('sliderNext')

  if (sliderPrev) sliderPrev.addEventListener('click', () => changeSlide(currentSlide - 1))
  if (sliderNext) sliderNext.addEventListener('click', () => changeSlide(currentSlide + 1))
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => changeSlide(index))
  })

  startSlider()
}

function startSlider() {
  stopSlider()
  slideInterval = setInterval(() => {
    changeSlide(currentSlide + 1)
  }, 3000)
}

function stopSlider() {
  clearInterval(slideInterval)
}

function changeSlide(n) {
  if (slides.length === 0) return
  
  slides[currentSlide].classList.remove('active')
  if (dots[currentSlide]) dots[currentSlide].classList.remove('active')

  currentSlide = (n + slides.length) % slides.length
  
  slides[currentSlide].classList.add('active')
  if (dots[currentSlide]) dots[currentSlide].classList.add('active')
  
  startSlider()
}

document.addEventListener('DOMContentLoaded', initPage)

